﻿using System;
using System.Collections.Concurrent;
using System.Collections.Generic;
using System.Collections.Specialized;
using System.Linq;
using System.Text;

using Newtonsoft.Json;

public class StudyService
{
    private const string STUDY_TYPE = "IMAGE";

    private readonly AuthService _authService;
    private readonly ConcurrentDictionary<string, StationConfiguration> _stationConfigMap;
    private readonly PatientImageRepository _patientImageRepository;
    private readonly PatientRepository _patientRepository;

    public StudyService(
        ConcurrentDictionary<string, StationConfiguration> stationConfigMap)
    {
        _stationConfigMap = stationConfigMap;

        _patientImageRepository = new PatientImageRepository();

        _patientRepository = new PatientRepository();

        _authService = new AuthService();
    }

    public void InsertUpdatePatientStudies(
        string patientIcn, 
        int patientId)
    {
        var station = _patientRepository.GetStationNumberForPatient(patientId);

        IReadOnlyList<VixStudy> studies = GetPatientStudies(patientIcn, patientId, station);

        _patientImageRepository.InsertUpdateStudies(studies);
    }

    private IReadOnlyList<VixStudy> GetPatientStudies(
        string patientIcn,
        int patientId,
        string station)
    {
        if (_stationConfigMap.ContainsKey(station))
        {
            var config = _stationConfigMap[station];

            var requestBytes = GetRequestAsBytes(patientIcn, station);

            try
            {
                var headers = GetRequestHeaders(config, station);
                if (headers != null)
                {
                    var response = WebRequestor.Post(config.StudyQueryUrl, requestBytes, "get-studies-for-patient", headers);

                    return GetStudiesFromResponse(response, patientId);
                }
            }
            catch (Exception)
            {
                throw new Exception("Services Unavailable");
            }
        }

        return new List<VixStudy>();
    }

    private byte[] GetRequestAsBytes(string patientIcn, string station)
    {
        var request = new StudyQueryRequest { patientICN = patientIcn, siteNumber = station };

        var json = JsonConvert.SerializeObject(request);

        return Encoding.UTF8.GetBytes(json);
    }

    private NameValueCollection GetRequestHeaders(StationConfiguration config, string station)
    {
        var vc = new VistACredentials { Access = config.Access, Verify = config.Verify };

        var tokenForSite = _authService.GetUserToken(config.GetUserTokenUrl, vc);
        if (!string.IsNullOrEmpty(tokenForSite))
            return new NameValueCollection
                {
                    { "xxx-sitename", config.SiteName },
                    { "xxx-userToken", tokenForSite },
                    { "xxx-ssn", config.Ssn },
                    { "xxx-duz", config.Duz },
                    { "xxx-sitenumber", station },
                    { "xxx-fullname", config.FullName }
                };

        return null;
    }

    private IReadOnlyList<VixStudy> GetStudiesFromResponse(string response, int patientId)
    {
        var result = new List<VixStudy>();

        var results = JsonConvert.DeserializeObject<VixStudies>(response);
        if (results != null && results.Studies != null && results.Studies.Any())
        {
            var imageStudies = results.Studies.Where(s => s.StudyType.Equals(STUDY_TYPE, StringComparison.OrdinalIgnoreCase));
            var count = imageStudies.Count();
            if (count > 0)
            {
                foreach (var study in imageStudies)
                {
                    study.PatientId = patientId;
                    result.Add(study);
                }
            }
        }

        return result;
    }

}
